<?php

namespace App\Models;

use Illuminate\Notifications\Notifiable;
use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Carbon\Carbon;

class User extends Authenticatable
{
    use Notifiable;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'first_name', 'last_name', 'birthdate',
        'phone', 'gender', 'password', 'region_id', 'district_id', 'address'
    ];

    /**
     * full name attribute
     *
     * @param string
    */
    public function getFullNameAttribute()
    {
        return $this->last_name.' '.$this->first_name;
    }

     /**
     * Automatically set the birthdate attribute to 
     * NULL when it is an empty string
     * 
     * @param string $value
     */
    public function setBirthdateAttribute($value)
    {		
        if (empty($value) ) {
            $this->attributes['birthdate'] = NULL;
        } else {
            $this->attributes['birthdate'] = Carbon::createFromFormat('d-m-Y',$value);
        }
    }

    /**
     * Get birthdate with format
     * NULL when it is an empty string
     * 
     * @param string $value
     */
    public function getBirthdateAttribute($value)
    {
        if ( empty($value) ) {
            return '';
        } 

        return $this->asDateTime($value)->format('d-m-Y');
    }

	public function getGenderFormatAttribute($value)
	{
		return $value == 1 ? 'Мужской' : 'Женский';
	}



    /**
     * The attributes that should be hidden for arrays.
     *
     * @var array
     */
    protected $hidden = [
        'password', 'remember_token',
    ];

    /**
     * The attributes that should be cast to native types.
     *
     * @var array
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
    ];

	public function orders()
	{
		return $this->hasMany(Order::class, 'user_id', 'id');
	}

    public function roles()
    {
    	return $this->belongsToMany(Role::class, 'role_user', 'user_id', 'role_id');
    }

	public function hasAnyRole($roles)
    {
        if (is_array($roles)) {
            $has_roles = cache()->remember('USER-'.$this->id.'-ROLES', now()->addMinutes(60), function() use ($roles) {
                return $this->roles;
            });

            if ($has_roles->whereIn('code', $roles)->count() > 0) {
                return true;
            }
        }
        return false;
        
    }

    public function hasRole($role) 
    {
        if ($this->hasAnyRole([$role])) {
            return true;
        } 
        return false;
    }

}
