<?php

namespace App\Models;

use Upload; 
use App\Models\Category;
use App\Models\OrderProduct;
use Illuminate\Database\Eloquent\Model;

class Product extends Model
{
    protected $guarded = [];
	protected $appends = ['url'];

	public function scopeSpareParts($query, $value)
	{
		if (in_array(1,$value) && in_array(2, $value)) {
			$query->where('to_1', 1)->where('to_2', 2);
		} elseif (in_array(1,$value)) {
			$query->where('to_1', 1);
		} elseif (in_array(2,$value)) {
			$query->where('to_2',1);
		}
	}

	public function scopeQuantity($query) 
	{
		$query->where('quantity', '>', 0);
	}

    public function getNameAttribute()
    {
        return $this->{'name_'.app()->getLocale()} ?? $this->name_ru;
    }

    public function getShortDescriptionAttribute()
    {
        return $this->{'short_description_'.app()->getLocale()};
    }

    public function getDescriptionAttribute()
    {
        return $this->{'description_'.app()->getLocale()};
    }

    public function getMetaTitleAttribute()
    {
        return $this->{'meta_title_'.app()->getLocale()};
    }

    public function getMetaDescriptionAttribute()
    {
        return $this->{'meta_description_'.app()->getLocale()};
    }

    public function getPriceFormatAttribute()
	{		
		return number_format($this->price, 0, ',', ' ');
    }

	public function getDiscountPriceFormatAttribute()
	{	    
	    return number_format($this->price * (1 - $this->discount/100), 0, ',', ' ');
	}

	public function getCurrentPriceAttribute()
	{
		if ($this->discount > 0) {
			return $this->price * (1 - $this->discount/100);
		}
		return $this->price;
	}	
    
    public function comments()
	{
		return $this->hasMany(Comment::class, 'product_id', 'id');
	}

    public function manufacturer()
    {
        return $this->hasOne(Manufacturer::class, 'id', 'manufacturer_id');
    }

    public function category()
	{
		return $this->hasOne(Category::class, 'id', 'category_id');
	}

    public function categories()
    {
	    return $this->belongsToMany(Category::class, 'product_category');
    }
    
    public function cars()
    {
		return $this->belongsToMany(Car::class, 'product_car');
    }

    public function manufacturers()
    {
		return $this->belongsToMany(Manufacturer::class, 'product_manufacturer');
    }
    
    public function getRatingAttribute()
	{
		return round($this->comments()->where('status', 1)->avg('rating'));
    }
    
    public function getCountCommentAttribute()
    {
        return $this->comments()->where('status', 1)->count();
    }

    public function getIsFavoriteAttribute()
	{
		return collect(json_decode(request()->cookie('user_favorite_list', '[]')))->contains($this->id);
	}

	public function getImageAttribute()
	{
		if (Upload::hasFile('product', $this->id)) {
			return Upload::getFile('product', $this->id);
		}
		return '';
	}

	public function scopeCategorized($query, Category $category=null)
	{
		if (is_null($category)) {
			return $query->with('categories');
		}

		$category_ids = $category->getDescendantsAndSelf()->pluck('id');

		return $query->with('categories')
			->join('product_category', 'product_category.product_id', 'products.id')
			->whereIn('product_category.category_id', $category_ids);
	}

	public function getUrlAttribute()
	{
		return route('frontend.product.show', $this->alias);
	}

	public function orderProducts()
	{
		return $this->hasMany(OrderProduct::class, 'product_id', 'id');
	}

	public function paymeSetting()
	{
		return $this->hasOne(PaymeSetting::class, 'id', 'payme_setting_id');
	}
}
