<?php

namespace App\Http\Controllers;

use App\Models\Role;
use App\Models\User;
use App\Rules\Phone;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use App\Library\PhoneVerification;
use Illuminate\Support\Facades\DB;

class UserController extends Controller
{
    /**
    * user login page
    */
    public function getLogin()
    {      
    	return view('frontend.user.login');
    }

    /**
    * Authentication on this form.
    * @param request()
    * @return $data
    */
    public function postLogin()
    {
        if (auth()->check()) {
            return redirect()->back();
        }

        $data = request()->only('phone', 'password');
        $data['phone'] = preg_replace('/[\+\-\(\)]/', '', $data['phone']);

        $rules = [
            'phone' => ['required', 'exists:users,phone', 'digits:12', new Phone],
            'password' => 'required',
        ];

        $remember_me = request()->has('remember') ? true : false;

        $validator = validator()->make($data, $rules);

        if ($validator->fails()) {  
            session()->flash('error', trans('auth.login.holder'));
            return redirect()->back()->withInput();
        }
        else {
            $attempt = auth()->attempt(['phone' => $data['phone'], 'password' => $data['password']], $remember_me);

            if ($attempt) {
                return redirect()->intended(route('home'));
            }
            session()->flash('error', trans('auth.login.holder'));
            return redirect()->back()->withInput();

        }
    }

    /**
    * user register page
    */

    public function getRegister()
    {        
        return view('frontend.user.register');
    }
    
    /**
     * Store a new user.
     *
     * @return \Illuminate\Http\Response
     */
    public function postRegister()
    {
    	request()->merge([
            'phone' => preg_replace('/[\+\-\(\)\s]/', '', request()->phone)
        ]);

    	request()->validate([
            'first_name' => 'required|max:255',
            'last_name' => 'required|max:255',
            'phone' => ['required', 'unique:users,phone', 'digits:12', new Phone],
            'password'   => 'required|confirmed'
        ]);

    	$user = User::create([
            'first_name'      => request()->first_name,
            'last_name' => request()->last_name,
    		'phone'     => request()->phone,
            'password'  => bcrypt(request()->password),
            'phone_verified_at' => now()
        ]);
        
        if ($user) {
			auth()->guard()->login($user);
			$role = Role::where('code', 'client')->first();
			$user->roles()->attach($role->id);
			$code = rand(111111, 999999);
			$message = 'Ваш код для подтверждения: '.$code.' shop.driversvillage.uz';
			// PhoneVerification::storeCode($user, $code, $message);            			
        }
    	// return redirect()->route('frontend.verify')->with('status', 'Мы отправили вам код активации. Проверьте свою электронную почту и нажмите на ссылку, чтобы подтвердить.');
        return redirect()->route('home');
    }

    /**
     * Verify the last sent code Time
     * 
     * @param  string $phone
     * @return array|boolean
     */
    private function verifyLastCodeTime($phone)
    {
    	$last_code = SmsCode::where('phone', $phone)->latest()->first();

        if ($last_code) {
            $add_minutes = $last_code->created_at->addMinutes(1);
            $diff_second = Carbon::now()->diffInSeconds($add_minutes);

            if ($diff_second < 60) {
            	$this->errors = [
                    'errors' => [
                    	'limit' => [
                    		trans('user.number_second', ['second' => $diff_second])
                    	]
                    ]
                ];
                return false;
            }
        }
        return true;
    }

    /**
     * Check sms activation code
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function checkCode(Request $request)
    {
        $phone = preg_replace('/[\+\-\(\)]/is', '', $request->phone);

        $code_check = SmsCode::where('phone', $phone)
            ->orderBy('id', 'desc')
            ->where('code',  $request->code)
            ->where('expires_at', '>=', Carbon::now())
            ->first();

        if ($code_check) {
            $response = [
                'error' => 0
            ];
        } else {
            $response = [
                'error' => 1
            ];
        }
        return response()->json($response);
    }

    /**
    * Logout.
    */
    public function logout()
    {
        auth()->logout();
        return redirect()->intended(route('home'));
    }

    public function resetPassword()
    {      
    	return view('frontend.user.reset');
    }

	public function resetPasswordForm()
	{
		request()->merge([
            'phone' => preg_replace("/[\s\+\(\)\-]/", '', request()->phone),
        ]);
        $data = request()->validate([
            'phone' => ['required', 'exists:users,phone', 'digits:12', new Phone],
            'type' => 'required|in:phone,code',
        ]);
		$user = User::where('phone', $data['phone'])->first();
        if ($data['type'] == 'phone') {
            DB::beginTransaction();            
            if ($user) {
                try {
                    $code = rand(111111, 999999);
                    $message = 'Ваш код для восстановления пароля: '.$code.' shop.driversvillage.uz';
                    $verify = PhoneVerification::storeCode($user, $code, $message);
                    $time_left = $this->timeLeft($verify);
                    DB::commit();
                    return response()->json([
						'type' => 'phone',
                        'status' => true,
                        'time_left' => $time_left,
                    ]);
                } catch (\Exception $e) {
                    DB::rollback();
                    return response()->json([
                        'message' => 'error',
                        'errors' => [
                            'phone' => [
                                'Номер телефона не найден',
                            ],
                        ],
                    ], 422);
                }
            }
        } elseif ($data['type'] == 'code') {            
            DB::beginTransaction();
            if ($user) {
                request()->validate([
                    'code' => [
                        'required',
                        Rule::exists('sms_codes')->where(function ($query) use ($user) {
                            $query->where('user_id', $user->id);
                            $query->where('code', request()->code);
                            $query->where('expires_at', '>=', now());
                            $query->orderBy('id', 'desc');
                        }),
                    ],
                ], [], [
                    'code' => 'Код',
                ]);
                try {
                    $new_password = Str::random(8);
                    $message = 'Ваш новый пароль для личного кабинета shop.driversvillage.uz: '.$new_password;
                    if (config('app.env') == 'production') {
                        PhoneVerification::sendSms($user->phone, $message);
                    }
                    $user->password = bcrypt($new_password);
                    $user->phone_verified_at = now();
                    $user->save();
                    DB::commit();
                    return response()->json([
						'type' => 'code',
                        'message' => 'Ваш пароль отправлен на ваш номер телефона',
                        'url' => route('frontend.user.getLogin'),
                    ], 200);
                } catch (\Exception $e) {
                    info($e->getMessage());
                    DB::rollback();
                    return response()->json([
                        'message' => 'error',
                        'errors' => [
                            'code' => [
                                'Попробуйте позже',
                            ],
                        ],
                    ], 422);
                }
            }
        }
	}

	public function resetPasswordSendCode($user)
	{		
		$code = rand(111111, 999999);
		$message = 'Ваш код для восстановления пароля: '.$code.' shop.driversvillage.uz';
        $sms_code = PhoneVerification::sendSms($phone, $user, $message);
        return response()->json([
            'time_left' => $this->timeLeft($sms_code),
            'success' => true
        ]);
	}

	public function timeLeft($sms_code)
    {
        if ($sms_code) {
            $time_left = strtotime($sms_code->expires_at) - strtotime(now());

            if ($time_left < 1) {
                return 0;
            }

            return $time_left;

        } else {
            return 0;
        }

    }

}
