<?php

namespace App\Http\Controllers;

use Cart;
use Session;
use Validator;
use App\Models\Car;
use App\Library\Helper;
use App\Models\Comment;
use App\Models\Product;
use App\Models\Category;
use App\Models\Favorite;
use App\Models\Manufacturer;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cookie;

class ProductController extends Controller
{
	// Cookie stored times in minutes
	protected $cookie_lifetime = 525600;
	protected $favorite_cookie = 'user_favorite_list';
    
	public function index(Request $request)
    {
		$params = func_get_args();
		unset($params[0]);
		$categories = $this->categories($params);
        if (!$categories['last']) {
            abort(404);
        }
		$category = $categories['last'];		
		$products = $this->filter($request, $category);
		$breadcrumbs = Helper::renderBreadcrumbs($categories['breadcrumbs']);
		$manufacturers = $this->manufacturers($category);
		$cars = $this->cars($category);		
		$inputs = $request->all();
		$parent_categories = Category::whereNull('parent_id')->get();
    	return view('frontend.product.index', compact('breadcrumbs', 'category', 'params', 'parent_categories', 'products', 'manufacturers', 'cars', 'inputs'));
    }

	/**
     * Get all categories by url params
     * 
     * @param  array $params
     * @return array
     */
    private function categories($params)
    {
		if (in_array('all', $params)) {
			$breadcrumbs[] = [
                'name' => __('main.all_category'),
                'url' => route('frontend.product.index', 'all')
            ];
			return [
				'breadcrumbs' => $breadcrumbs,
				'last' => 'all'
			];
		}
        $placeholders = implode(',', array_fill(0, count($params), '?'));
        $categories = Category::whereIn('alias', $params)
            ->orderByRaw("field(alias,{$placeholders})", $params)
            ->get();

        $breadcrumbs = [];
        foreach ($categories as $category) {
            $breadcrumbs[] = [
                'name' => $category->name,
                'url' => $category->url
            ];
        }

        return [
            'breadcrumbs' => $breadcrumbs,
            'last' => $categories->last()
        ];
    }

	public function postFilter(Request $request)
	{						
		$products = $this->filter($request);		
        return response()->json([
            'view' => view()->make('partials.products', compact('products'))
                            ->render(),
            'pagination' => $products->appends(array_filter($request->except('categories')))
                                ->withPath('')
                                ->links()
                                ->toHtml(),
            'total' => $products->total()
        ]);
	}

	/**
     * Get products
     * 
     * @param  Object $request
     * @param  array  $params
     * 
     * @return Object
     */
    private function filter($request, $category = [])
    {
		if ($category == 'all') {
			$categories = 'all';
		} else {
			$categories = $request->input('categories');
			if ($categories && in_array('all', $categories)) {
				$categories = 'all';
			} elseif(empty($categories)) {
				$categories = [$category->id];
			}
		}
        $manufacturers = $request->input('manufacturers');
        $cars = $request->input('cars');
        $products = Product::quantity()->has('paymeSetting')->with('categories')
            ->select('products.*')
			->when($categories != 'all', function($query) use ($categories) {
				return $query->join('product_category', 'product_category.product_id', '=', 'products.id')
            		->where('product_category.category_id', end($categories));
			})
            ->when($request->has('sort_by'), function($query) use ($request) {
				if ($request->input('sort_by') == 'price_asc') {
					return $query->orderByRaw('CASE WHEN `discount` > 0 THEN `price` * (1 - `discount`/100) ELSE `price` END ASC');
				} elseif ($request->input('sort_by') == 'price_desc') {
					return $query->orderByRaw('CASE WHEN `discount` > 0 THEN `price` * (1 - `discount`/100) ELSE `price` END DESC');
				} elseif ($request->input('sort_by') == 'rating') {
					return $query->orderByRaw('(select round(avg(rating), 0) from comments where product_id = products.id and status = 1) DESC');
				}
            })->when($request->has('spare_part'), function($query) use ($request) {
				$query->spareParts($request->spare_part);
			})
			->when($request->has('price_min') && !is_null($request->price_min), function($query) use ($request) {
                $price_min = str_replace(' ', '', $request->price_min);
                return $query->where('price', '>=', $price_min);

            })->when($request->has('price_max') && !is_null($request->price_max), function($query) use ($request) {
                $price_max = str_replace(' ', '', $request->price_max);
                return $query->where('price', '<=', $price_max);
            })->when(!is_null($manufacturers) && count($manufacturers) > 0, function($query) use ($manufacturers) {
                return $query->whereIn('manufacturer_id', $manufacturers);
            })->when(!is_null($cars) && count($cars) > 0, function($query) use ($cars) {
				return $query->whereHas('cars', function($query) use ($cars) {
					$query->whereIn('id', $cars);
				});                
            });			

    	return $products->paginate(15);
    }

	/**
     * Get brands by url params
     * 
     * @param  Object $category
     * @return array
     */
    private function manufacturers($category)
    {		
		$products = $category != 'all' ? Product::categorized($category)
					->groupBy('manufacturer_id')
					->pluck('manufacturer_id') : [];		
        $manufacturers = Manufacturer::
			when($category != 'all', function($query) use ($products) {
				$query->whereIn('id', $products);
			})->orderBy('name', 'asc')
            ->get();
        $manufacturers->map(function($item) {
            $item->count = $item->products()->count();
            return $item;
        });

        return $manufacturers;
    }

	/**
     * Get cars by url params
     * 
     * @param  Object $category
     * @return array
     */
    private function cars($category)
    {				
		$cars = Car::when($category != 'all', function($query) use ($category) {
			$query->whereHas('products', function($query) use ($category) {
				$query->categorized($category);
			});
		})->get();
        $cars->map(function($item) {
            $item->count = $item->products()->count();
            return $item;
        });

        return $cars;
    }

    public function show($alias)
    {
		$product = Product::has('paymeSetting')->where('alias', $alias)->firstOrFail();
		$others = Product::has('paymeSetting')->whereHas('categories', function($query) use ($product) {
			$query->whereIn('id', $product->categories()->whereNotNull('parent_id')->pluck('id'));
		})->where('id', '!=', $product->id)->get();
		$breadcrumbs = [];
        if ($product->categories) {
            foreach ($product->categories as $category) {
                $breadcrumbs[] = [
                    'name' => $category->name,
                    'url' => $category->url
                ];
            }
        }
        $breadcrumbs = Helper::renderBreadcrumbs($breadcrumbs);
		$open_comment = false;
		if (auth()->check()) {
			$has_ordered = auth()->user()->orders()->whereHas('products', function($query) use ($product) {
				$query->where('product_id', $product->id);
			})->where('status', 'delivered')->first();
			if ($has_ordered) {
				$open_comment = true;
			}
		}
		$product_cart = Cart::get($product->id);
    	return view('frontend.product.show', compact('product', 'open_comment', 'product_cart', 'others', 'breadcrumbs'));
    }

    public function manufacturer($alias)
    {
		$manufacturer = Manufacturer::where('alias', $alias)->firstOrFail();
     	return view('frontend.product.car' , compact('manufacturer'));
    }

    public function category($alias)
    {		
		$manufacturer = Manufacturer::where('alias', $alias)->firstOrFail();
		$products = $manufacturer->products()->pluck('id');		
		// $products = $manufacturer->products()->has('paymeSetting')->pluck('id');		
		// $products = Product::has('paymeSetting')->where('manufacturer_id', $manufacturer->id)->pluck('id');
		$categories = Category::whereHas('products', function($query) use ($products) {
			$query->whereIn('id', $products);
		})->get();		
	
        return view('frontend.product.category', compact('categories', 'manufacturer'));
    }

	public function favorite()
	{
		$list = $this->getFavoriteList();
		$products = Product::has('paymeSetting')->whereIn('id', $list)->paginate(18);
		return view('frontend.page.favorite', compact('products'));
	}

	public function getFavoriteList()
	{
		$list = request()->cookie($this->favorite_cookie, '[]');
		return collect(json_decode($list));
	}

	public function syncFavorite()
	{
		$data = request()->validate([
			'id' => 'required',
		]);

		$list = $this->getFavoriteList();

		if ($list->contains($data['id'])) {
			$list = $list->reject(function ($value, $key) use ($data) {
				return $value == $data['id'];
			});
			$status = 'false';
			$message = __("main.fav_removed");
		} else {
			$list->prepend($data['id']);
			$status = 'true';
			$message = __("main.fav_added");
		}


		$new_list = json_encode($list);
		Cookie::queue($this->favorite_cookie, $new_list, $this->cookie_lifetime);
		return response()->json([
			'total' => $list->count(),
			'status' => $status,
			"message" => $message
		]);
	}

	public function search()
	{
		$input = trim(strip_tags(request()->input('q', '')));
		$products = Product::has('paymeSetting')->where(function($query) use ($input) {
			$query->where('name_ru', 'like', '%'.$input.'%')
				->orWhere('name_uz', 'like', '%'.$input.'%');
		})->get();
        return response()->json([
            'view' => view()->make('partials.search-result', compact('products'))->render()
        ]);

	}

	public function comment($product_id)
	{
		$product = Product::has('paymeSetting')->findOrFail($product_id);
		if (auth()->check()) {
			$has_ordered = auth()->user()->orders()->whereHas('products', function($query) use ($product) {
				$query->where('product_id', $product->id);
			})->where('status', 'delivered')->first();
			if (!$has_ordered) {
				abort(404);
			}
		} else {
			abort(404);
		}
		$data = request()->only('rating', 'review');

        $data['user_id'] = request()->user()->id;
        $data['product_id'] = $product_id;

        $rules = [
            'review' => 'nullable|max:1000'
        ];
        $validator = Validator::make($data, $rules);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()]);
        }
        else{
            Session::flash('product_comment', 'success');
            $create = Comment::create($data);
            return response()->json(['success' => trans('Спасибо! Ваш комментарий успешно добавлен.')]);
        }
	}
}
