<?php

namespace App\Http\Controllers;

use Hash;
use Upload;
use App\Models\Car;
use App\Models\MyCar;
use App\Models\Order;
use App\Models\Region;
use App\Models\Address;
use App\Models\Favorite;
use App\Models\Manufacturer;
use App\Models\Notification;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use App\Http\Resources\MyCarResource;
use App\Http\Resources\OrderResource;
use App\Http\Resources\NotificationResource;
use App\Http\Resources\Address as AddressResource;
use App\Http\Resources\Favorite as FavoriteResource;

class CabinetController extends Controller
{
	protected $user;
	
	public function __construct()
    {
        $this->middleware(function ($request, $next) {
            $this->user = $request->user();
            return $next($request);
        });
    }

    public function index()
    {        
		$user = $this->user;
		return view('frontend.cabinet.main', [
			'user' => $user
		]);
    }
    
    public function profile()
    {		
     	$user = $this->user;
		$regions = Region::whereNull('parent_id')->pluck('name_'.app()->getLocale(), 'id');
		$districts = Region::where('parent_id', $user->region_id)->pluck('name_'.app()->getLocale(), 'id');
        return response()->json([
			'user' => $user,
			'regions' => $regions,
			'districts' => $districts
		]);
    }

	public function getDistricts()
	{
		$districts = Region::where('parent_id', request()->id)->pluck('name_'.app()->getLocale(), 'id');
		return $districts;
	}

    public function updateProfile()
    {
        $user = $this->user;
		
        $data = request()->only([
            'first_name',
            'last_name',            
            'gender',
            'birthdate',        
            'old_password',
			'region_id',
			'district_id',
			'address',
            'password',
            'password_confirmation'
        ]);        

        $rules = [
            'first_name' => 'required|max:255',
            'last_name'  => 'required|max:255',
            'birthdate'  => 'nullable|date_format:d-m-Y',
            'gender'     => 'boolean',
            'password'   => 'min:6|confirmed',
			'region_id' => ['nullable', Rule::exists('regions', 'id')->where('parent_id', null)],
			'district_id' => ['nullable', Rule::exists('regions', 'id')->where('parent_id', request()->region_id)],
			'address' => 'nullable|max:200',
        ];
        $validator = validator()->make($data, $rules);
		
        if ($validator->fails()) {
            return response()->json(['errors' => $validator->messages()]);
        }

        /**
         * Save new password
        */
        if (request()->has('password') && request()->has('old_password')) {
            if (!Hash::check($data['old_password'], $user->password)) {
                return response()->json(['errors' => ['old_password' => trans('auth.login.no_password')]]);
            }
            $data['password'] = bcrypt($data['password']);
        } else {
            $data = array_except($data, ['password']);            
        }
		
        $user->update($data);

        return response()->json([
            'status' => 1,
            'message' => trans('cabinet.profile.success_message')
        ]);
    }

    public function address()
    {
        $user = request()->user();
        $items = Address::where('user_id', $user->id)->get();
        $addresses = AddressResource::collection($items);
        return response()->json($addresses);
    }

    public function region()
    {
        $regions = Region::where('parent_id', null)->get();
        return response()->json($regions);
    }

    public function district()
    {
        $region_id = request()->region_id;
        $districts = Region::where('parent_id', $region_id)->pluck('name_ru', 'id');
        return response()->json($districts);
    }

    public function createAddress()
    {
        $user = request()->user();

        $data = request()->only([
            'street',
            'house_number',            
            'apartment',
            'entrance',        
            'floor',
            'region_id',
            'district_id',
            'main_address'
        ]);

        $rules = [
            'street' => 'required|max:255',
            'house_number'  => 'required|max:10',
            'apartment'  => 'nullable|max:10',
            'entrance'     => 'nullable|max:10',
            'floor'   => 'nullable|max:10',
            'region_id' => 'required|integer',
            'district_id' => 'required|integer',
            'main_address' => 'boolean'
        ];

        $validator = validator()->make($data, $rules);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->messages()]);
        }
        $data['user_id'] = $user->id; 
        $address = Address::create($data);

        return response()->json([
            'status' => 1
        ]);
    }

    public function editAddress($id)
    {
        $user = request()->user();        
        $address = Address::find($id);
        if ($address) {
            if ($address->user_id != $user->id) {
                return response()->json([
                    'status' => 0
                ]);
            } else {
                return response()->json([
                    'status' => 1,
                    'address' => $address
                ]);
            }
        } else {
            return response()->json([
                'status' => 0
            ]);
        }               
    }
    
    public function updateAddress($id)
    {        
        $user = request()->user();

        $data = request()->only([
            'street',
            'house_number',            
            'apartment',
            'entrance',        
            'floor',
            'region_id',
            'district_id',
            'main_address'
        ]);

        $rules = [
            'street' => 'required|max:255',
            'house_number'  => 'required|max:10',
            'apartment'  => 'nullable|max:10',
            'entrance'     => 'nullable|max:10',
            'floor'   => 'nullable|max:10',
            'region_id' => 'required|integer',
            'district_id' => 'required|integer',
            'main_address' => 'boolean'
        ];

        $validator = validator()->make($data, $rules);        

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->messages()]);
        }
        
        $address = Address::find($id);
        if ($address && $address->user_id = $user->id) {
            $address->update($data);
            return response()->json([
                'status' => 1
            ]);
        } else {
            return response()->json([
                'status' => 0
            ]);
        }

    }

    public function deleteAddress($id)
    {
        $address = Address::find($id);
        $address->delete();
        return response()->json(['status' => 1]);
    }

    public function changeMainAddress($id)
    {
        $user = request()->user();
        $address = Address::find($id);

        if ($address->main_address == 0) {
            $address->main_address = 1;
            $has_main_address = Address::where('user_id', $user->id)
                                        ->where('main_address', 1)
                                        ->first();
            if (!empty($has_main_address)) {
                $has_main_address->main_address = 0;
                $has_main_address->update();
            }
        } else {
            $address->main_address = 0;
        }

        $address = $address->update();

        return response()->json(['status' => 1]);
    }


    public function notifications()
    {
        $notifications = Notification::latest()->where(function($query) {
			$query->where('user_id', $this->user->id)->orWhere('phone', $this->user->phone);
		})->paginate(10);
		return NotificationResource::collection($notifications);
    }

	public function getOrders($type)
	{
		$orders = Order::latest()
					->with('products')
					->when($type == 'active', function($query) {
						$query->whereIn('status', ['pending', 'confirmed', 'paid', 'shipping']);
					})->when($type == 'completed', function($query) {
						$query->where('status', 'delivered');
					})->when($type == 'canceled', function($query) {
						$query->where('status', 'canceled');
					})
					->where('user_id', $this->user->id)
					->get();
		$orders->map(function($order){
            $order->products->map(function($item){
				if ($item->product) {
					$item->image = asset($item->product->image);
				}
            });
        });
        return response()->json(['orders' => OrderResource::collection($orders)]);
	}

	public function getActiveOrders()
	{
		$active_orders = Order::latest()
						->with('products')
						->whereIn('status', ['pending', 'confirmed', 'paid', 'shipping'])
						->where('user_id', $this->user->id)
						->get();
		$active_orders->map(function($order){
            $order->products->map(function($item){
				if ($item->product) {
					$item->image = asset($item->product->image);
				}
            });
        });

        return response()->json(['active_orders' => OrderResource::collection($active_orders)]);
	}

	public function getCompletionOrders()
	{
		$completion_orders = Order::latest()
							->with('products')
                            ->whereIn('status', ['delivered', 'canceled'])
                            ->where('user_id', $this->user->id)
                            ->get();

        $completion_orders->map(function($order){
            $order->products->map(function($item){
                if ($item->product) {
                    $item->image = asset($item->product->image);
                }                
            });
        });

        return response()->json(['completion_orders' => OrderResource::collection($completion_orders)]);
	}

	public function cars()
	{
		$cars = MyCar::where('user_id', $this->user->id)->paginate(10);
		return MyCarResource::collection($cars);
	}

	public function carForm($id = null)
	{
		$car = MyCar::where('user_id', $this->user->id)->where('id', $id)->first();
		return response()->json([
			'car' => $car,
			'mode' => $car ? 'edit' : 'create',
			'manufacturers' => Manufacturer::pluck('name', 'id'),
			'cars' => Car::pluck('name', 'id')
		]);		
	}

	public function postCarForm($id = null)
	{	
		$data = request()->validate([
			'manufacturer_id' => 'required|exists:manufacturers,id',
			'car_id' => 'required|exists:cars,id',
			'year' => 'required|digits:4|max:'.now()->format('Y'),
			'milage' => 'required|max:12|regex:/^-?[0-9]+(?:\.[0-9]{1,2})?$/'
		],[],[
			'manufacturer_id' => 'Марка автомобиля',
			'car_id' => 'Модель автомобиля',
			'year' => 'Год выпуска',
			'milage' => 'Пробег'
		]);
		
		if ($id) {
			$car = MyCar::where('user_id', $this->user->id)->where('id', $id)->firstOrFail();
			$car->update($data);
		} else {
			$data['user_id'] = $this->user->id;
			$car = MyCar::create($data);
		}
		return response()->json([
            'status' => 1,
            'message' => $id ? 'Успешно изменено' : 'Успешно создано' 
        ]);
	}
}
