<?php

namespace App\Http\Controllers\Backend;

use DB;
use Upload;
use App\Models\Car;
use App\Library\Helper;
use App\Models\Product;
use App\Models\Category;
use App\Models\Manufacturer;
use App\Models\PaymeSetting;
use Illuminate\Http\Request;
use App\Imports\ProductImport;
use App\Exports\ProductExport;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Excel;

class ProductController extends Controller
{
    public function index()
    {
        return view('backend.product.index');
    }
    
    public function product_excel_export()
    {
        return Excel::download(new ProductExport(), 'Продукты.xlsx');
    }

    public function data()
   	{
   		$product = Product::withCount('orderProducts');
           return datatables()->of($product)
                ->editColumn('price', function($item) {
                    return $item->price_format;
                })->toJson();
   	}

   	public function getForm($id = null)
   	{
   		$product = Product::find($id);
        if (!$product) $product = [];
        $categories = Category::all()->toHierarchy();
        $manufacturers = Manufacturer::get();
        $cars = Car::get();
        $payme_settings = PaymeSetting::select(DB::raw("
			concat(name,'-',code) as title,
			id
		"))->pluck('title', 'id');
        return view('backend.product.form', [
            'product' => $product,
            'id' => $id,
            'categories' => $categories,
            'manufacturers' => $manufacturers,
            'cars' => $cars,
			'payme_settings' => $payme_settings
        ]);
   	}

   	public function postForm($id = null) 
   	{    
   		$data = request()->validate([
   			'name_ru'  				=> 'required|max:255',
            'name_uz'  				=> 'nullable|max:255',
   			'short_description_ru'  => 'required|max:255',
            'short_description_uz'  => 'nullable|max:255',
            'description_ru'  		=> 'required',
            'description_uz'  		=> 'nullable',
            'price'             	=> 'required|max:12|regex:/^-?[0-9]+(?:\.[0-9]{1,2})?$/',
            'discount'          	=> 'nullable|min:1|max:100',
            'meta_title_ru' 		=> 'max:255',
            'meta_title_uz' 		=> 'max:255',
            'category_id'   		=> 'nullable|integer|exists:categories,id',
            'manufacturer_id' 		=> 'nullable|integer|exists:manufacturers,id',
            'meta_description_ru' 	=> 'max:255',
            'meta_description_uz' 	=> 'max:255',
            'category'  			=> 'required',
            'video'                 => 'mimes:mp4|max:10000',
			'payme_setting_id'	    => 'nullable|exists:payme_settings,id'
   		]);

        if(request()->has('to_1') && request('to_1')) {
            $data['to_1'] = 1;
        } else {
            $data['to_1'] = 0;
        }

        if(request()->has('to_2') && request('to_2')) {
            $data['to_2'] = 1;
        } else {
            $data['to_2'] = 0;
        }


        unset($data['category']);
        unset($data['video']);

   		if ($id == null) {
   			$data['alias'] = Helper::alias(new Product, $data['name_ru']);
   			$product = Product::create($data);
   			session()->flash('success', trans('alert.success.insert'));
   		} else {
   			$product = Product::find($id);    
            $data['alias'] = Helper::alias(new Product, $data['name_ru']);
            $product->update($data);
            DB::table('product_car')->where('product_id', $product->id)->delete();
            session()->flash('success', trans('alert.success.update'));
        }

        $cat_ids = [];
        $old_cats = $product->categories->pluck('id')->toArray();
        
        foreach(request()->category as $cat_id) {
            if (!in_array($cat_id, $old_cats)) {
                $count = DB::table('product_category')
                    ->where('category_id', $cat_id)
                    ->count();

                $cat_ids[$cat_id] = [
                    'position' => $count
                ];
            } else {
                $cat_ids[] = $cat_id;
            }
        }

		if (!request()->has('category_id')) {
			$product->update([
				'category_id' =>array_key_first($cat_ids)
			]);
		}

        $product->categories()->sync($cat_ids);
        $product->cars()->sync(request()->cars);
        $product->manufacturers()->sync(request()->manufacturers);
        
        /**
         * Remove images
        */
        $removed_images = explode(',', request()->input('removed_images'));
        $images = Upload::getFiles('product', $product->id);

        if (request()->input('main_image', 0) != 0) {
            $images = Upload::getFiles('product', $product->id);
            Upload::swapFirst('product', $product->id, request()->input('main_image'));
        }

        if (request()->input('removed_images') != null && count($removed_images) > 0) {

            foreach($removed_images as $del) {
                $files[] = $images[$del];
            }

            $removed = Upload::removeFiles('product', $id, $files);
        }
        
        if (request()->hasFile('images')) {
            Upload::saveFiles('product', $product->id, request()->file('images'), [
                'name' => $product->alias
            ]);
        }

        if (request()->hasFile('video')) {
            Upload::saveFile('product_video', $product->id, request()->file('video'), ['name' =>  $product->alias]);
        }

   		return redirect()->route('backend.product.index');
   	}

   	public function delete()
   	{
   		if (request()->has('id')) {
            $item = Product::findOrFail(request()->input('id'));
            $item->categories()->sync([]);
            $item->cars()->sync([]);
            Upload::removeFiles('product', $item->id);
   			$item->delete();

   			$response = [
                'status' => 'success',
                'message' => trans('alert.success.delete')
            ]; 
   		} else {
   			$response = [
                'status' => 'error',
                'message' => 'Пожалуйста попробуйте снова'
            ];
   		}

   		return response()->json($response);
    }

    public function import(Request $request)
    {
        request()->validate([
            'products' => 'required|mimes:xlx,xls,xlsx|max:2048'
        ]);
        $items = Excel::toArray(new ProductImport, $request->file('products'));
        // unset($items[0][0]);
        // dd($items[0]);
        foreach ($items[0] as $row) {
            $product = Product::create([
                'name_ru'               =>  $row[0],
                'name_uz'               =>  $row[1],
                'short_description_ru'  =>  $row[2],
                'short_description_uz'  =>  $row[3],
                'description_ru'        =>  $row[4],
                'description_uz'        =>  $row[5],
                'price'                 =>  $row[6] ?? 0,
                'discount'              =>  $row[7] ?? NULL,
                'to_1'                  =>  $row[8] ?? 0,
                'to_2'                  =>  $row[9] ?? 0,
                'manufacturer_id'       =>  $row[11],
                'alias'                 =>  Helper::alias(new Product, $row[0]),
                'quantity'              =>  $row[13],
                'category_id'           =>  $row[11],
                'meta_title_ru'         =>  $row[0],
                'meta_title_uz'         =>  $row[1],
                'meta_description_ru'   =>  $row[2],
                'meta_description_uz'   =>  $row[3],
            ]);

            if(isset($row[10])) {
                $product->categories()->sync(explode('|', $row[10]));
            }
            

            if(isset($row[12])) {
            $product->cars()->sync(explode('|', $row[12])); 
            }
     
        }                    

        return back()->with('massage', 'User Imported Successfully');
    }
    
	public function changeQuantity()
	{
		$data = request()->validate([
			'id' => 'required|exists:products,id',
			'quantity' => 'required|integer|min:0'
		]);
		$product = Product::find($data['id']);
		$product->update([
			'quantity' => $data['quantity']
		]);
		return response()->json([
			'status' => true,
			'message' => trans('alert.success.delete')
		]);
	}
}
