<?php

namespace App\Http\Controllers\Backend;

use App\Models\Order;
use App\Models\Notification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\File;

class OrderController extends Controller
{
    public function index()
    {		
		$statuses = __('cabinet.order.statuses');
		$payment_types = __('cabinet.order.payment_types');
        return view('backend.order.index', compact('statuses','payment_types'));
    }

    public function data()
   	{
   		$query = Order::latest();
   		return datatables()->of($query)
			->filterColumn('full_name', function($query,$keyword) {
				$sql = "CONCAT(orders.last_name,' ',orders.first_name) like ?";
                $query->whereRaw($sql,  ["%{$keyword}%"]);
			})->rawColumns(['status_label_format'])->toJson();
   	}    

    public function view($id)
    {
        $order = Order::findOrFail($id);
		$can_change_status = $this->checkAccessForChange($order);
		$detail = null;
		if (in_array($order->payment_type, ['click', 'payme', 'uzum']) && $order->payment_id) {
			$detail = $order->payment;
		}

        return view('backend.order.view', compact('order', 'can_change_status', 'detail'));
    }

	public function changeStatus()
	{		
		$data = request()->validate([
			'order_id' => 'required|exists:orders,id',
			'status' => 'required|in:confirmed,paid,shipping,delivered,canceled'
		]);

		$order = Order::findOrFail($data['order_id']);
		$history = $order->histories()->create([
			'status' => $data['status'],
			'user_id' => auth()->user()->id
		]);
		$order->update([
			'status' => $data['status']
		]);
		$notification_messages = $this->getNotificationMessages($data['status'],$order->id);
		Notification::create([
			'user_id' => $order->user_id,
			'phone' => $order->phone_number,
			'text_ru' => $notification_messages[0],
			'text_uz' => $notification_messages[1],
		]);
		if ($data['status'] == 'canceled') {
			foreach ($order->products as $product) {
				$product->product->update([
					'quantity' => $product->product->quantity + $product->quantity
				]);
			}
		}
		session()->flash('success', trans('alert.success.update'));
		return redirect()->back();
	}

	public function getNotificationMessages($status, $order_id)
	{
		$messages_uz = [
			'confirmed' => "Sizning №:{$order_id} raqamli buyurtmangiz tasdiqlandi",
			'paid' => "Siz №:{$order_id} raqamli buyurtmaga muvaffaqqiyatli to‘lov qildingiz",
			'shipping' => "Sizning №:{$order_id} raqamli buyurtmangiz yetkazib berish jarayonida",
			'delivered' => "Sizning №:{$order_id} raqamli buyurtmangiz muvaffaqiyatli yetkazib berildi",
			'canceled' => "Sizning №:{$order_id} raqamli buyurtmangiz bekor qilindi",
		];
		$messages_ru = [
			'confirmed' => "Ваш заказ №:{$order_id} подтвержден",
			'paid' => "Вы успешно оплатили №:{$order_id} заказа",
			'shipping' => "Ваш заказ №:{$order_id} доставляется",
			'delivered' => "Ваш заказ №:{$order_id} успешно доставлен",
			'canceled' => "Ваш заказ №:{$order_id} отменен",
		];
		return [
			$messages_ru[$status],$messages_uz[$status]
		];
	}

	public function checkAccessForChange($order)
	{
		if (in_array($order->payment_type, ['payme', 'click', 'apelsin']) && in_array($order->status, ['paid', 'canceled'])) {
			return false;
		}
		return true;
	}
}
